import java.io.*;
import java.util.*;

public class PropsToDocs {

    private List _sections;

    public PropsToDocs(InputStream in) throws Exception {

        _sections = new ArrayList();

        parse(new BufferedReader(new InputStreamReader(in)));
    }

    private void parse(BufferedReader reader) throws Exception {

        String line = reader.readLine();

        Section section = null;
        Sub sub = null;
        while (line != null) {

            // parse through the header of the next section
            while (line != null && section == null) {

                if ( (line.length() > 2) && line.startsWith("# ") && !line.startsWith("#  ")) {

                    String title = line.substring(2).trim();
                    int i = title.indexOf("#");
                    if (i != -1) {
                        title = title.substring(0, i - 1).trim();
                    }
                    section = new Section(title);
                    _sections.add(section);
                    reader.readLine(); // skip ending ####
                }
                line = reader.readLine();
            }

            // parse through all subs for the section
            while (line != null && section != null) {

                line = line.trim();

                if ( (line.length() > 2) && line.startsWith("# ")) { // && !line.startsWith("#  ")) {

                    if (sub == null) {
                        sub = new Sub(); 
                        section.subs().add(sub);
                    }
                    sub.addDescriptionLine(line.substring(2));
                    line = reader.readLine();
                } else if (line.length() > 1 && !line.startsWith("#")) {
                    // defining something... scroll till not
                    while (line != null && line.length() > 1 && !line.startsWith("#")) {
                        sub.addDefinitionLine(line);
                        line = reader.readLine();
                        if (line != null) line = line.trim();
                    }
                    sub = null;
                } else if (line.startsWith("##")) {
                    // entering next section
                    section = null;
                    line = reader.readLine();
                } else {
                    // treat as blank line
                    if (line.length() == 0) {
                        sub = null;
                    }
                    line = reader.readLine();
                }
            }

        }
    }

    public void writeHTML(PrintWriter writer) throws Exception {

        writer.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        writer.println("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" ");
        writer.println("              \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">");
                      
        writer.println("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en-US\" lang=\"en-US\">");
        writer.println("<head>");
        writer.println("  <title>Configuration Reference</title>");
        writer.println("  <link rel=\"stylesheet\" type=\"text/css\" href=\"docstyle.css\" />");
        writer.println("</head>");

        writer.println("<body>");
        writer.println("<div id=\"header\">");
        writer.println("  <!-- <a href=\"../index.html\" id=\"logo\"></a> -->");
        writer.println("  <div id=\"title\">");
        writer.println("    <h1>Configuration Reference</h1>");
        writer.println("  </div>");
        writer.println("</div>");
       
        writer.println("<div id=\"toc\">");
        writer.println("  <h2>Table of Contents</h2>");
        writer.println("  <div id=\"tocbox\">");
        addTableOfContents(writer, "    ");
        writer.println("  </div>");
        writer.println("</div>");
        writer.println();
        addBody(writer);
        writer.println("</body>");
        writer.println("</html>");
        writer.flush();
    }

    public void addTableOfContents(PrintWriter writer, String p) throws Exception {

        writer.println(p+ "<ol>");
        for (int i = 0; i < _sections.size(); i++) {
            Section section = (Section) _sections.get(i);
            String sectionID = getID(section.getTitle());
            writer.println(p+ "  <li> <a href=\"#" + sectionID + "\">" + section.getTitle() + "</a></li>");
        }
        writer.println(p+ "</ol>");
    }

    public void addBody(PrintWriter writer) throws Exception {

        for (int i = 0; i < _sections.size(); i++) {
            Section section = (Section) _sections.get(i);
            String sectionID = getID(section.getTitle());
            int secNum = i + 1;

            writer.println("<div class=\"sec2\">");
            writer.println("  <h2><a name=\"" + sectionID + "\">" + secNum + ". " + section.getTitle() + "</a></h2>");
            writer.println("  <ul>");
            for (int j = 0; j < section.subs().size(); j++) {
                Sub sub = (Sub) section.subs().get(j);
                writer.println("    <li> <code><b>" + sub.getDefinitionText() + "</b></code>");
                writer.println("      <p>");
                writer.println("        " + sub.getDescriptionText());
                writer.println("      </p><p>&nbsp;</p>");
                writer.println("    </li>");
            }
            writer.println("  </ul>");
            writer.println("</div>");
        }
    }

    private static String getID(String s) {
        StringBuffer buf = new StringBuffer();
        s = s.toLowerCase();
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (c >= 'a' && c <= 'z') {
                buf.append(c);
            }
        }
        return buf.toString();
    }

    public static void main(String[] args) {

        try {
            PropsToDocs p = new PropsToDocs(new FileInputStream(new File(args[0])));
            p.writeHTML(new PrintWriter(System.out));
        } catch (Throwable th) {
            th.printStackTrace();
            System.exit(1);
        }
    }

    public class Section {

        private String _title;
        private List _subs;

        public Section(String title) {

            _title = title;
            _subs = new ArrayList();
        }

        public String getTitle() {

            return _title;
        }

        public List subs() {

            return _subs;
        }

    }

    public class Sub {

        private String _title;
        private StringBuffer _descBuffer;
        private String _firstDescriptionLine;
        private List _defs;

        public Sub() {

            _descBuffer = new StringBuffer();
            _defs = new ArrayList();
        }

        public void setTitle(String title) {

            _title = title;
        }

        public String getTitle() {

            return _title;
        }

        public void addDescriptionLine(String line) {

            if (_firstDescriptionLine == null) {
                _firstDescriptionLine = line;
            }

            _descBuffer.append(line + "\n");
        }

        public void addDefinitionLine(String line) {

            String[] parts = line.split("=");
            if (parts.length < 2) {
                throw new RuntimeException("Unable to parse definition line: '" + line + "' (expected one '=' inside)");
            }
            _defs.add(parts[0].trim());
        }

        public String getDescriptionText() {

            return _descBuffer.toString();
        }

        public String getDefinitionText() {

            if (_defs.size() == 0) return _firstDescriptionLine;
            if (_defs.size() == 1) return (String) _defs.get(0);

            String s = getCommonEnding(_defs);
            if (s.length() == 0 || s.indexOf(".") == -1) {
                throw new RuntimeException("No common ending in definition group containing '" + _defs.get(0) + "'");
            }

            StringBuffer z = new StringBuffer();
            z.append("*");
            String[] parts = s.split("\\.");
            for (int i = 1; i < parts.length; i++) {
                z.append("." + parts[i]);
            }
            return z.toString();
        }

        private String getCommonEnding(List strings) {

            StringBuffer out = new StringBuffer();

            // find shortest
            int shortestLength = Integer.MAX_VALUE;
            int shortestIndex = -1;
            for (int i = 0; i < strings.size(); i++) {
                String s = (String) strings.get(i);
                if (s.length() < shortestLength) {
                    shortestIndex = i;
                    shortestLength = s.length();
                }
            }

            for (int charIndex = 0; charIndex < shortestLength; charIndex++) {
                String s = (String) strings.get(0);
                char c = s.charAt(s.length() - charIndex - 1);
                for (int i = 1; i < strings.size(); i++) {
                    String n = (String) strings.get(i);
                    int ni = n.length() - charIndex - 1;
                    if (n.charAt(ni) != c) {
                        return out.toString();
                    }
                }
                out.insert(0, c);
            }

            return out.toString();
        }

    }

}
